/** @file   shadowtable.cpp
 * @brief   Implementation of ShadowTable - class.
 * @version $Revision: 1.5 $
 * @author  Tomi Lamminsaari
 */

#include "shadowtable.h"
#include "www_map.h"
#include "GfxManager.h"
#include "gfxid.h"
#include "www_assert.h"
#include "tileid.h"
#include "eng2d.h"
using namespace eng2d;
using std::list;
namespace WeWantWar {


/** Constructor.
 */
ShadowTable::ShadowItem::ShadowItem() :
  iShadowX( 0 ),
  iShadowY( 0 ),
  iShadowIndex( 0 )
{
}

/** Constructor. 
 */
ShadowTable::ShadowItem::ShadowItem(int aShadowX, int aShadowY, int aShadowIndex) :
  iShadowX( 32*aShadowX ),
  iShadowY( 32*aShadowY ),
  iShadowIndex( aShadowIndex )
{
}

/** Constructor
 */
ShadowTable::ShadowTable( const Rect2D& aArea ) :
  iArea( aArea ),
  iAlpha( 200 )
{
  // We construct the shadow items based on the data we found from the shadow
  // layer of the map.
  if ( Map::selectLayer( Map::EDynamicShadows ) < 0 ) {
    return;
  }
  
  if ( iArea.bottomright.intX() >= Map::getWidth( Map::IN_PIXELS ) ) {
    iArea.bottomright.vx = Map::getWidth( Map::IN_PIXELS ) - 1;
  }
  if ( iArea.bottomright.intY() >= Map::getHeight( Map::IN_PIXELS ) ) {
    iArea.bottomright.vy = Map::getHeight( Map::IN_PIXELS ) - 1;
  }
  
  for ( float x = iArea.topleft.vx; x < iArea.bottomright.vx; x += 32 ) {
    for ( float y = iArea.topleft.vy; y < iArea.bottomright.vy; y += 32 ) {
      Map::selectLayer( Map::EDynamicShadows );
      int tileX = static_cast<int>( x+1 ) / 32;
      int tileY = static_cast<int>( y+1 ) / 32;
      if ( Map::blockNumberAt( tileX, tileY, Map::IN_BLOCKS ) != 0 ) {
        Map::Block* tile = Map::blockAt( tileX, tileY, Map::IN_BLOCKS );
        ShadowItem* shadItem = new ShadowItem;
        shadItem->iShadowX = tileX * 32;
        shadItem->iShadowY = tileY * 32;
        shadItem->iShadowIndex = tile->user4;
        iShadows.push_back( shadItem );
        
      } else {
        // This is a hack that makes it easier to build tree shadows.
        Map::selectLayer( Map::EDecorative );
        Map::Block* tile = Map::blockAt( tileX, tileY, Map::IN_BLOCKS );
        if ( tile->user4 == TileID::EHighTree ) {
          this->createHighTreeShadow( tileX, tileY );
        } else if ( tile->user4 == TileID::EMediumTree ) {
          this->createMediumTreeShadow( tileX, tileY );
        }
      } 
    }
  }
  Map::selectLayer( Map::EBackground );
}



/** Destructor
 */
ShadowTable::~ShadowTable()
{
  this->clear();
}



/** Sets the alpha level
 */
void ShadowTable::setAlpha( int aAlpha )
{
  iAlpha = aAlpha;
}



/** Applies the shadows
 */
void ShadowTable::applyShadows( BITMAP* aTarget )
{
  GraphicsContainer* gfxObj = GfxManager::findGfxContainer( GfxId::KDynamicShadows );
  WWW_ASSERT( gfxObj != 0 );
  
  set_multiply_blender( 0,0,0, iAlpha );
  list< ShadowItem* >::iterator shadowIter = iShadows.begin();
  while ( shadowIter != iShadows.end() ) {
    int shaX = (*shadowIter)->iShadowX - Map::scrollX;
    int shaY = (*shadowIter)->iShadowY - Map::scrollY;
    draw_trans_rle_sprite( aTarget, gfxObj->GetRleSprite( (*shadowIter)->iShadowIndex ),
                           shaX, shaY );
                           
    shadowIter++;
  }
}



/** Clears the contents
 */
void ShadowTable::clear()
{
  list< ShadowItem* >::iterator shadowIter = iShadows.begin();
  while ( shadowIter != iShadows.end() ) {
    delete (*shadowIter);
    shadowIter++;
  }
  iShadows.clear();
}



///
/// Getter methods
/// ==============

/** Returns the area
 */
Rect2D ShadowTable::area() const
{
  return iArea;
}



/** Returns the alpha level
 */
int ShadowTable::alpha() const
{
  return iAlpha;
}



/** Creates the shadow for high tree.
 */
void ShadowTable::createHighTreeShadow(int aCenterX, int aCenterY )
{
  iShadows.push_back( new ShadowItem(aCenterX-1,aCenterY+0, 0) );
  
  iShadows.push_back( new ShadowItem(aCenterX-2, aCenterY+1, 0) );
  iShadows.push_back( new ShadowItem(aCenterX-1, aCenterY+1, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-0, aCenterY+1, 3) );
  
  iShadows.push_back( new ShadowItem(aCenterX-4, aCenterY+2, 10) );
  iShadows.push_back( new ShadowItem(aCenterX-3, aCenterY+2, 11) );
  iShadows.push_back( new ShadowItem(aCenterX-2, aCenterY+2, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-1, aCenterY+2, 3) );
  
  iShadows.push_back( new ShadowItem(aCenterX-5, aCenterY+3, 12) );
  iShadows.push_back( new ShadowItem(aCenterX-4, aCenterY+3, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-3, aCenterY+3, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-2, aCenterY+3, 17) );
  
  iShadows.push_back( new ShadowItem(aCenterX-5, aCenterY+4, 13) );
  iShadows.push_back( new ShadowItem(aCenterX-4, aCenterY+4, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-3, aCenterY+4, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-2, aCenterY+4, 16) );
  
  iShadows.push_back( new ShadowItem(aCenterX-4, aCenterY+5, 14) );
  iShadows.push_back( new ShadowItem(aCenterX-3, aCenterY+5, 15) );
}



/** Creates the shadow for medium high tree
 */
void ShadowTable::createMediumTreeShadow( int aCenterX, int aCenterY )
{
  iShadows.push_back( new ShadowItem(aCenterX-2,aCenterY+0, 10) );
  iShadows.push_back( new ShadowItem(aCenterX-1,aCenterY+0, 11) );
  
  iShadows.push_back( new ShadowItem(aCenterX-3,aCenterY+1, 12) );
  iShadows.push_back( new ShadowItem(aCenterX-2,aCenterY+1, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-1,aCenterY+1, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-0,aCenterY+1, 17) );
  
  iShadows.push_back( new ShadowItem(aCenterX-3,aCenterY+2, 13) );
  iShadows.push_back( new ShadowItem(aCenterX-2,aCenterY+2, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-1,aCenterY+2, 5) );
  iShadows.push_back( new ShadowItem(aCenterX-0,aCenterY+2, 16) );
  
  iShadows.push_back( new ShadowItem(aCenterX-2,aCenterY+3, 14) );
  iShadows.push_back( new ShadowItem(aCenterX-1,aCenterY+3, 15) );
}

} // end of namespace
